<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Models\Country;
use Illuminate\Http\Request;

class SettingsController extends Controller
{
    public function index()
    {
        $settings = Setting::all()->keyBy('key');
        $countries = Country::whereNotNull('currency')
                           ->select('id', 'name', 'currency', 'currency_symbol', 'currency_name')
                           ->orderBy('name')
                           ->get();
        
        return view('admin.settings.index', compact('settings', 'countries'));
    }

    public function update(Request $request)
    {
        $request->validate([
            'default_currency' => 'required|exists:countries,id',
        ]);

        // Get the selected country's currency details
        $selectedCountry = Country::find($request->default_currency);
        
        // Update currency settings
        Setting::setValue('default_currency', $selectedCountry->currency, 'string', 'Default currency code');
        Setting::setValue('default_currency_symbol', $selectedCountry->currency_symbol, 'string', 'Default currency symbol');
        Setting::setValue('default_currency_name', $selectedCountry->currency_name, 'string', 'Default currency name');
        Setting::setValue('default_currency_country_id', $selectedCountry->id, 'integer', 'Default currency country ID');

        return redirect()->route('admin.currency-settings.index')
            ->with('success', 'Currency settings updated successfully.');
    }

    /**
     * Show Payment Gateway Settings page
     */
    public function paymentGateways()
    {
        $settings = Setting::all()->keyBy('key');
        $gateway = request()->get('gateway', 'paypal'); // Default to PayPal
        
        return view('admin.settings.payment-gateways', compact('settings', 'gateway'));
    }

    /**
     * Update Payment Gateway Settings
     */
    public function updatePaymentGateways(Request $request)
    {
        $gateway = $request->input('gateway');
        
        // Validate based on gateway type
        $rules = [];
        $messages = [];
        
        switch ($gateway) {
            case 'paypal':
                $rules = [
                    'paypal_account' => 'nullable|email',
                    'paypal_client_id' => 'nullable|string',
                    'paypal_secret' => 'nullable|string',
                    'paypal_mode' => 'nullable|in:sandbox,live',
                    'paypal_active' => 'nullable|in:yes,no',
                ];
                break;
                
            case 'stripe':
                $rules = [
                    'stripe_key' => 'nullable|string',
                    'stripe_secret' => 'nullable|string',
                    'stripe_mode' => 'nullable|in:test,live',
                    'stripe_active' => 'nullable|in:yes,no',
                ];
                break;
                
            case 'paystack':
                $rules = [
                    'paystack_public_key' => 'nullable|string',
                    'paystack_secret_key' => 'nullable|string',
                    'paystack_merchant_email' => 'nullable|email',
                    'paystack_mode' => 'nullable|in:test,live',
                    'paystack_active' => 'nullable|in:yes,no',
                ];
                break;
        }
        
        $request->validate($rules, $messages);
        
        // Update PayPal settings
        if ($gateway === 'paypal') {
            Setting::setValue('paypal_account', $request->input('paypal_account', ''), 'string', 'PayPal Account Email');
            Setting::setValue('paypal_client_id', $request->input('paypal_client_id', ''), 'string', 'PayPal Client ID');
            Setting::setValue('paypal_secret', $request->input('paypal_secret', ''), 'string', 'PayPal Secret Key');
            Setting::setValue('paypal_mode', $request->input('paypal_mode', 'sandbox'), 'string', 'PayPal Mode (sandbox/live)');
            Setting::setValue('paypal_active', $request->input('paypal_active', 'no'), 'string', 'PayPal Active Status');
        }
        
        // Update Stripe settings
        if ($gateway === 'stripe') {
            Setting::setValue('stripe_key', $request->input('stripe_key', ''), 'string', 'Stripe Public Key');
            Setting::setValue('stripe_secret', $request->input('stripe_secret', ''), 'string', 'Stripe Secret Key');
            Setting::setValue('stripe_mode', $request->input('stripe_mode', 'test'), 'string', 'Stripe Mode (test/live)');
            Setting::setValue('stripe_active', $request->input('stripe_active', 'no'), 'string', 'Stripe Active Status');
        }
        
        // Update Paystack settings
        if ($gateway === 'paystack') {
            Setting::setValue('paystack_public_key', $request->input('paystack_public_key', ''), 'string', 'Paystack Public Key');
            Setting::setValue('paystack_secret_key', $request->input('paystack_secret_key', ''), 'string', 'Paystack Secret Key');
            Setting::setValue('paystack_merchant_email', $request->input('paystack_merchant_email', ''), 'string', 'Paystack Merchant Email');
            Setting::setValue('paystack_mode', $request->input('paystack_mode', 'test'), 'string', 'Paystack Mode (test/live)');
            Setting::setValue('paystack_active', $request->input('paystack_active', 'no'), 'string', 'Paystack Active Status');
        }
        
        return redirect()->route('admin.payment-gateways.index', ['gateway' => $gateway])
            ->with('success', ucfirst($gateway) . ' settings updated successfully.');
    }
}
